<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Role;
use App\Models\Department;
use App\Models\Provider;
use App\Models\Chief;
use App\Models\SubChief;
use App\Models\Employee;
use Illuminate\Support\Facades\Hash;
use App\Enum\UserRole;

class DemoUsersSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // 1. Create Departments
        $policeDept = Department::firstOrCreate(
            ['name' => 'Mega City Police Department'],
            [
                'city' => 'Mega City',
                'state' => 'NY',
                'contact_number' => '555-0111',
                'email' => 'contact@mcpd.gov',
                'address' => '100 Justice Way',
                'zip_code' => '10001'
            ]
        );

        $medicalDept = Department::firstOrCreate(
            ['name' => 'City General Hospital'],
            [
                'city' => 'Mega City',
                'state' => 'NY',
                'contact_number' => '555-0222',
                'email' => 'info@cityhospital.org',
                'address' => '500 Health Blvd',
                'zip_code' => '10002'
            ]
        );

        // 2. Create Roles
        $chiefRole = Role::firstOrCreate(['name' => UserRole::Chief->value]);
        $subChiefRole = Role::firstOrCreate(['name' => UserRole::SubChief->value]);
        $employeeRole = Role::firstOrCreate(['name' => UserRole::Employee->value]);
        $providerRole = Role::firstOrCreate(['name' => UserRole::Provider->value]);

        // --- Demo Chief (Supervisor ID: 0) ---
        $chiefUser = User::firstOrCreate(
            ['email' => 'gordon@mcpd.com'],
            [
                'name' => 'Gordon Freeman',
                'username' => 'chief_gordon',
                'password' => Hash::make('password'),
                'role' => UserRole::Chief->value,
                'phone' => '555-0100',
                'department_id' => $policeDept->id,
                'supervisor_id' => 0,
                'gender' => 'Male',
                'is_active' => true,
            ]
        );

        $chiefProfile = Chief::firstOrCreate(
            ['user_id' => $chiefUser->id],
            [
                'badge_number' => 'CH-001',
                'rank' => 'Police Chief',
                'appointed_date' => now()->subYears(5),
            ]
        );

        // Assign Role
        if (!$chiefUser->roles()->where('role_id', $chiefRole->id)->exists()) {
            $chiefUser->roles()->attach($chiefRole->id, [
                'entity_id' => $chiefProfile->id,
                'entity_type' => Chief::class,
                'entity_role' => 'Chief',
                'default_role' => true
            ]);
        }

        // --- Demo SubChief (Supervisor ID: Chief's ID) ---
        $subChiefUser = User::firstOrCreate(
            ['email' => 'barney@mcpd.com'],
            [
                'name' => 'Barney Calhoun',
                'username' => 'lt_barney',
                'password' => Hash::make('password'),
                'role' => UserRole::SubChief->value,
                'phone' => '555-0102',
                'department_id' => $policeDept->id,
                'supervisor_id' => $chiefUser->id,
                'gender' => 'Male',
                'is_active' => true,
            ]
        );

        $subChiefProfile = SubChief::firstOrCreate(
            ['user_id' => $subChiefUser->id],
            [
                'badge_number' => 'LT-104',
                'rank' => 'Lieutenant',
            ]
        );

        if (!$subChiefUser->roles()->where('role_id', $subChiefRole->id)->exists()) {
            $subChiefUser->roles()->attach($subChiefRole->id, [
                'entity_id' => $subChiefProfile->id,
                'entity_type' => SubChief::class,
                'entity_role' => 'SubChief',
                'default_role' => true
            ]);
        }

        // --- Demo Employees (Officers) (Supervisor ID: SubChief's ID) ---
        $employees = [
            ['name' => 'John Doe', 'badge' => 'OFF-201', 'rank' => 'Officer', 'username' => 'off_john'],
            ['name' => 'Jane Smith', 'badge' => 'OFF-202', 'rank' => 'Officer', 'username' => 'off_jane'],
            ['name' => 'Alex Murphy', 'badge' => 'OFF-999', 'rank' => 'Cyborg Officer', 'username' => 'robocop'],
        ];

        foreach ($employees as $empData) {
            $empUser = User::firstOrCreate(
                ['email' => $empData['username'] . '@mcpd.com'],
                [
                    'name' => $empData['name'],
                    'username' => $empData['username'],
                    'password' => Hash::make('password'),
                    'role' => UserRole::Employee->value,
                    'phone' => '555-1000',
                    'department_id' => $policeDept->id,
                    'supervisor_id' => $subChiefUser->id,
                    'is_active' => true,
                ]
            );

            $empProfile = Employee::firstOrCreate(
                ['user_id' => $empUser->id],
                [
                    'badge_number' => $empData['badge'],
                    'rank' => $empData['rank'],
                    'position' => 'Patrol',
                    'hired_date' => now()->subMonths(6),
                ]
            );

            if (!$empUser->roles()->where('role_id', $employeeRole->id)->exists()) {
                $empUser->roles()->attach($employeeRole->id, [
                    'entity_id' => $empProfile->id,
                    'entity_type' => Employee::class,
                    'entity_role' => 'Employee',
                    'default_role' => true
                ]);
            }
        }

        // --- Demo Provider (Medical) ---
        $providerUser = User::firstOrCreate(
            ['email' => 'alyx@hospital.com'],
            [
                'name' => 'Dr. Alyx Vance',
                'username' => 'dr_alyx',
                'password' => Hash::make('password'),
                'role' => UserRole::Provider->value,
                'phone' => '555-5000',
                'department_id' => $medicalDept->id,
                'supervisor_id' => 0,
                'gender' => 'Female',
                'is_active' => true,
            ]
        );

        $providerProfile = Provider::firstOrCreate(
            ['user_id' => $providerUser->id],
            [
                'npi' => '1234567890',
                'license_number' => 'MD-55555',
                'specialization' => 'Trauma Surgeon',
            ]
        );

        if (!$providerUser->roles()->where('role_id', $providerRole->id)->exists()) {
            $providerUser->roles()->attach($providerRole->id, [
                'entity_id' => $providerProfile->id,
                'entity_type' => Provider::class,
                'entity_role' => 'Provider',
                'default_role' => true
            ]);
        }
    }
}
