<?php

namespace App\Services;

use App\Enum\UserRole;
use App\Models\User;
use App\Mail\WelcomeUserMail;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Hash;
use App\Services\ImageService;
use App\Services\RoleService;

class UserService
{
    protected $imageService;
    protected $roleService;

    public function __construct(ImageService $imageService, RoleService $roleService)
    {
        $this->imageService = $imageService;
        $this->roleService = $roleService;
    }

    public function createUser(array $data): User
    {
        return DB::transaction(function () use ($data) {

            $date_of_birth = $data['date_of_birth'] ?? null;
            // $age = $date_of_birth ? Carbon::parse($date_of_birth)->age : null;
            
            // Determine if using default password
            $isDefaultPassword = !isset($data['password']);
            $plainPassword = $isDefaultPassword ? 'Password123' : $data['password'];
            
            // Handle Profile Image Upload
            $profileImagePath = null;
            if (!empty($data['profile_image'])) {
                $profileImagePath = $this->imageService->saveBase64Image($data['profile_image']);
            }

            $user = User::create([
                'name' => trim(($data['first_name'] ?? '') . ' ' . ($data['last_name'] ?? '')) ?: ($data['name'] ?? 'Unknown'),
                'username' => $data['username'] ?? null,
                'email' => $data['email'] ?? null,
                'password' => Hash::make($plainPassword),
                'role' => $data['role'] ?? 'employee', // Default to employee
                'phone' => $data['phone'] ?? null,
                'address' => $data['address'] ?? null,
                'city' => $data['city'] ?? null,
                'state' => $data['state'] ?? null,
                'zip_code' => $data['zip_code'] ?? null, // Map postal_code -> zip_code
                'gender' => $data['gender'] ?? null,
                'date_of_birth' => $data['date_of_birth'] ?? null,
                'is_active' => $data['is_active'] ?? true,
                'profile_image' => $profileImagePath,
                'department_id' => $data['department_id'] ?? null,
                'supervisor_id' => $data['supervisor_id'] ?? 0,
            ]);

            // Assign Roles
            if (!empty($data['role'])) {
                $this->roleService->syncUserRoles($user, [$data['role']]);
            }

            // Send welcome email logic (Commented out until Mail configuration is set)
            /*
            if ($isDefaultPassword && $user->email) {
                try {
                    Mail::to($user->email)->send(new WelcomeUserMail($user, $plainPassword));
                } catch (\Exception $e) {
                    \Log::error('Failed to send welcome email: ' . $e->getMessage());
                }
            }
            */

            return $user;
        });
    }

    public function updateUser(User $user, array $data): User
    {
        return DB::transaction(function () use ($user, $data) {
            
            // Handle Image Update
            if (isset($data['profile_image'])) {
                // If it's a base64 string, save new one
                if (preg_match('/^data:image\/(\w+);base64,/', $data['profile_image'])) {
                    $data['profile_image'] = $this->imageService->saveBase64Image($data['profile_image']);
                } else {
                    // If it's URL or null, keep existing or clear?
                    // Usually we keep existing if null isn't explicitly requested to clear
                     unset($data['profile_image']); 
                }
            }

            $user->update([
                'name' => $data['name'] ?? $user->name,
                'email' => $data['email'] ?? $user->email,
                'username' => $data['username'] ?? $user->username,
                'phone' => $data['phone'] ?? $user->phone,
                'address' => $data['address'] ?? $user->address,
                'city' => $data['city'] ?? $user->city,
                'state' => $data['state'] ?? $user->state,
                'zip_code' => $data['zip_code'] ?? $user->zip_code,
                'gender' => $data['gender'] ?? $user->gender,
                'date_of_birth' => $data['date_of_birth'] ?? $user->date_of_birth,
                'is_active' => $data['is_active'] ?? $user->is_active,
                // Only update profile_image if processed above
                'profile_image' => $data['profile_image'] ?? $user->profile_image, 
                'department_id' => $data['department_id'] ?? $user->department_id,
                'supervisor_id' => $data['supervisor_id'] ?? $user->supervisor_id,
            ]);

            // Sync Roles if provided
            if (!empty($data['role'])) {
                 // Check if role changed
                 if ($user->role !== $data['role']) {
                     $user->update(['role' => $data['role']]);
                     $this->roleService->syncUserRoles($user, [$data['role']]);
                 }
            }

            // Return user with processed image URL for frontend (Base64 or URL)
            // Actually, we usually return a URL in API, but pathways requests base64 return? 
            // The prompt says: "return base64 to frontend as we are doing in pathways".
            
            return $user;
        });
    }
}
