<?php

namespace App\Services;

use App\Enum\UserRole;
use App\Models\User;
use App\Models\Role;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class RoleService
{
    public function assignRoleToUser(User $user, string $roleName, ?Model $entity = null)
    {
        $role = Role::where('name', $roleName)->firstOrFail();

        $hasDefaultRole = DB::table('user_roles')
            ->where('user_id', $user->id)
            ->where('default_role', true)
            ->exists();

        // Prepare pivot data
        $pivotData = [
            'default_role' => !$hasDefaultRole,
            'created_at'  => now(),
            'updated_at'  => now(),
        ];

        if ($entity) {
            $pivotData['entity_id'] = $entity->id;
            $pivotData['entity_type'] = get_class($entity);
            $pivotData['entity_role'] = ucfirst($roleName);
        }

        $user->roles()->syncWithoutDetaching([
            $role->id => $pivotData
        ]);

        // Ensure the user_role column is updated to reflect the assigned role (Primary Role concept)
        // If user already has a role set, we might not want to overwrite it unless logic dictates
        if (!$user->role || $user->role === 'employee') { 
             $user->update(['role' => $roleName]);
        }
    }

    public function syncUserRoles(User $user, array $roleNames): void
    {
        foreach ($roleNames as $roleName) {
            
            if (is_array($roleName)) {
                $roleName = $roleName['name'] ?? $roleName['value'] ?? null;
            }

            if (!is_string($roleName)) {
                continue; 
            }

            $roleEnum = UserRole::tryFrom($roleName);

            if (!$roleEnum) {
                // If not in Enum, maybe check if it's a valid role string anyway?
                // For now strict enum check
                continue;
            }

            $entity = null;
            
            // Auto-create/find entity profile if it doesn't exist?
            // Pathways does this: "$modelClass::firstOrCreate(['user_id' => $user->id])"
            // We need to be careful about required fields (e.g. city_id, practice_id).
            // For now, we will try to find existing or create empty if allowed.
            if ($modelClass = $roleEnum->modelClass()) {
                 // First Call specific: Entity tables usually require `practice_id` or similar 
                 // which might not be available here. 
                 // However, following Pathways logic:
                 try {
                    $entity = $modelClass::firstOrCreate(
                        ['user_id' => $user->id]
                        // We might need formatted default data here or it will fail if strict
                    );
                 } catch (\Exception $e) {
                     // If creation fails due to missing strict columns, we proceed without linking entity yet
                     \Log::warning("Could not auto-create entity profile for role $roleName: " . $e->getMessage());
                     $entity = null;
                 }
            }

            $role = Role::where('name', $roleEnum->value)->first();
            
            if (!$role) continue;

            $exists = DB::table('user_roles')
                ->where('user_id', $user->id)
                ->where('role_id', $role->id)
                ->exists();

            if (! $exists) {
                $this->assignRoleToUser(
                    user: $user,
                    roleName: $roleEnum->value,
                    entity: $entity
                );
            }
        }
    }
}
