<?php

namespace App\Services;

use Illuminate\Support\Facades\File;

class ImageService
{
    /**
     * Convert an image file to base64 with mime type
     *
     * @param string $imagePath
     * @return array|null
     */
    public function base64Convertor(string $imagePath): ?array
    {
        // Case 1: It's a URL
        if (filter_var($imagePath, FILTER_VALIDATE_URL)) {
            $image = @file_get_contents($imagePath);
            if (!$image) {
                return null;
            }

            $mimeType = finfo_buffer(finfo_open(FILEINFO_MIME_TYPE), $image);
            return [
                'file' => base64_encode($image),
                'mime_type' => $mimeType,
            ];
        }

        // Case 2: It's a storage file
        // Check both direct path and storage/app path
        $fullPath = storage_path('app/' . $imagePath);
        if (!File::exists($fullPath)) {
             // Fallback to checking if it's already a full path or public path
             if (File::exists($imagePath)) {
                 $fullPath = $imagePath;
             } else {
                 return null;
             }
        }

        $image = File::get($fullPath);
        $mimeType = File::mimeType($fullPath);

        return [
            'file' => base64_encode($image),
            'mime_type' => $mimeType,
        ];
    }

    /**
     * Get full data URI string for profile image
     *
     * @param string|null $imagePath
     * @return string|null
     */
    public function getImageDataUri(?string $imagePath): ?string
    {
        if (!$imagePath) {
            return null;
        }

        // If explicitly a URL (avatar placeholder), return as is
        if (filter_var($imagePath, FILTER_VALIDATE_URL)) {
             return $imagePath;
        }

        $base64 = $this->base64Convertor($imagePath);
        return $base64 ? "data:{$base64['mime_type']};base64,{$base64['file']}" : null;
    }

    public function saveBase64Image(string $base64Image, string $folder = 'profile-images'): ?string
    {
        try {
            // Handle data URI scheme if present
            if (preg_match('/^data:image\/(\w+);base64,/', $base64Image, $type)) {
                $base64Image = substr($base64Image, strpos($base64Image, ',') + 1);
                $extension = strtolower($type[1]); // jpg, png, gif
            } else {
                // assume jpg/png if no header, but risky. Let's try to detect or default to jpg
                $extension = 'jpg';
            }

            $base64Image = str_replace(' ', '+', $base64Image);
            $imageData = base64_decode($base64Image);

            if ($imageData === false) {
                return null;
            }

            $fileName = uniqid() . '.' . $extension;
            $filePath = $folder . '/' . $fileName;

            // Save to storage/app/{folder}
            $fullPath = storage_path('app/' . ltrim($filePath, '/'));
            
            if (!file_exists(dirname($fullPath))) {
                mkdir(dirname($fullPath), 0777, true);
            }

            file_put_contents($fullPath, $imageData);

            return $filePath;
        } catch (\Exception $e) {
            \Log::error('Image save failed: ' . $e->getMessage());
            return null;
        }
    }
}
