<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasFactory, Notifiable, HasApiTokens;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'username',
        'role',
        'password',
        'phone', 
        'address', 
        'city', 
        'state', 
        'zip_code', 
        'gender', 
        'date_of_birth', 
        'is_active', 
        'profile_image',
        'department_id',
        'supervisor_id'
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
        ];
    }

    /**
     * Get the department that the user belongs to.
     */
    public function department()
    {
        return $this->belongsTo(Department::class);
    }

    /**
     * Get the supervisor of the user.
     */
    public function supervisor()
    {
        return $this->belongsTo(User::class, 'supervisor_id');
    }

    /**
     * Get the subordinates of the user.
     */
    public function subordinates()
    {
        return $this->hasMany(User::class, 'supervisor_id');
    }

    // Role Relationship
    public function roles()
    {
        return $this->belongsToMany(Role::class, 'user_roles')
                    ->withPivot('entity_id', 'entity_type', 'entity_role', 'default_role')
                    ->withTimestamps();
    }

    // Specialized Profiles (Remaining)
    public function admin()
    {
        return $this->hasOne(Admin::class);
    }

    public function provider()
    {
        return $this->hasOne(Provider::class);
    }

    public function chief()
    {
        return $this->hasOne(Chief::class);
    }

    public function subChief()
    {
        return $this->hasOne(SubChief::class);
    }

    public function employee()
    {
        return $this->hasOne(Employee::class);
    }

    // Helper to check roles
    public function hasRole(string $roleName): bool
    {
        return $this->roles()->where('name', $roleName)->exists();
    }
}
