<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Appointment extends Model
{
    use HasFactory;
    
    protected $fillable = [
        'employee_id', // Changed from patient_id
        'cpt_code',
        'fees',
        'status',
        'provider_id',
        'appointment_date',
        'start_time',
        'end_time',
        'type',
        'title',
        'description',
        'color_primary',
        'color_secondary',
        'all_day',
        'resizable_before_start',
        'resizable_after_end',
        'draggable',
    ];

    protected array $datetimeFields = [ "start_time", "end_time"];

    public function provider()
    {
        return $this->belongsTo(Provider::class);
    }

    public function employee() // Changed method name
    {
        return $this->belongsTo(Employee::class);
    }
    
    // Note: Other relationships like invoice, note, modifiers skipped for now as they don't exist in first_call yet

    /**
     * Hook model events
     */
    protected static function booted()
    {
        // Before save → convert local → UTC
        static::saving(function ($model) {
            $model->convertLocalToUtc();
        });

        // After find → convert UTC → local
        static::retrieved(function ($model) {
            $model->convertUtcToLocal();
        });
    }

    /**
     * Convert Local time to UTC before saving
     * 
     * $timeoffset = minutes from UTC
     * 
     * Formula: Local + offset = UTC
     *     → addMinutes($offsetMinutes) converts local → UTC correctly for ALL zones
     */
    protected function convertLocalToUtc(): void
    {
        // timeoffset is minutes from UTC
        $offsetMinutes = (int) request()->query('timeoffset', 0);

        foreach ($this->datetimeFields as $field) {
            if (empty($this->{$field})) {
                continue;
            }

            // Combine date and time to create a full datetime object
            // If field is just time, Carbon assumes today's date, which is fine, 
            // but for start_time, better to us appointment_date if possible to handle day rollovers theoretically,
            // though strictly "start_time" is just H:i:s.
            // Using logic from pathways:
            $localDateTime = $this->appointment_date . ' ' . $this->{$field};

            $utcDateTime = Carbon::parse($localDateTime)->addMinutes($offsetMinutes);

            $this->{$field} = $utcDateTime->format('H:i:s');

            if ($field === 'start_time') {
                $this->appointment_date = $utcDateTime->format('Y-m-d');
            }
        }
    }

    /**
     * Convert stored UTC time back to user's local timezone when reading
     */
    protected function convertUtcToLocal(): void
    {
        $offsetMinutes = (int) request()->query('timeoffset', 0);

        foreach ($this->datetimeFields as $field) {
            if (empty($this->{$field})) {
                continue;
            }

            $utcDateTime = $this->appointment_date . ' ' . $this->{$field};
            $localDateTime = Carbon::parse($utcDateTime)->addMinutes(-$offsetMinutes);

            $this->{$field} = $localDateTime->format('H:i'); // return as H:i for frontend inputs usually

            if ($field === 'start_time') {
                $this->appointment_date = $localDateTime->format('Y-m-d');
            }
        }
    }
}
