<?php

namespace App\Http\Resources;

use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;
use App\Services\ImageService;
use Carbon\Carbon;

class UserResource extends JsonResource
{
    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        // $this refers to the USER model instance here
        
        $imageService = new ImageService();
        $date_of_birth = $this->date_of_birth ?? null;
        $age = $date_of_birth ? Carbon::parse($date_of_birth)->age : null;
        
        // Handle Profile Image data URI conversion
        $profile_image = $this->profile_image ? $imageService->getImageDataUri($this->profile_image) : null;

        $data = [
            'id' => $this->id,
            'name' => $this->name,
            'username' => $this->username,
            'email' => $this->email,
            'role' => $this->role, // Primary role string
            'phone' => $this->phone,
            'address' => $this->address,
            'city' => $this->city,
            'state' => $this->state,
            'zip_code' => $this->zip_code,
            'gender' => $this->gender,
            'date_of_birth' => $date_of_birth,
            'age' => $age,
            'is_active' => (bool) $this->is_active,
            'profile_image' => $profile_image,
            'department_id' => $this->department_id,
            'supervisor_id' => $this->supervisor_id,
            'department' => new DepartmentResource($this->whenLoaded('department')),
            'supervisor' => new UserResource($this->whenLoaded('supervisor')),
            'created_at' => $this->created_at,
            'updated_at' => $this->updated_at,
            
            // Basic role relations (only IDs/basic info usually, or nested if requested)
            // 'roles' => $this->roles, 
        ];

        // Conditional loading of role-specific profiles
        // We check if the relation is loaded to avoid N+1 issues, or check if specific profile exists
        if ($this->relationLoaded('admin') && $this->admin) {
             $data['admin_profile'] = $this->admin;
        }
        
        if ($this->relationLoaded('chief') && $this->chief) {
             $data['chief_profile'] = $this->chief;
             // Could format badge/rank here if needed flat
             $data['rank'] = $this->chief->rank;
             $data['badge_number'] = $this->chief->badge_number;
             $data['appointed_date'] = $this->chief->appointed_date;
        }
        
        if ($this->relationLoaded('subChief') && $this->subChief) {
             $data['subchief_profile'] = $this->subChief;
             $data['rank'] = $this->subChief->rank;
             $data['badge_number'] = $this->subChief->badge_number;
        }

        if ($this->relationLoaded('employee') && $this->employee) {
             $data['employee_profile'] = $this->employee;
             $data['rank'] = $this->employee->rank;
             $data['badge_number'] = $this->employee->badge_number;
             $data['position'] = $this->employee->position;
             $data['hired_date'] = $this->employee->hired_date;
        }

        if ($this->relationLoaded('provider') && $this->provider) {
             $data['provider_profile'] = $this->provider;
             $data['npi'] = $this->provider->npi;
             $data['license_number'] = $this->provider->license_number;
             $data['specialization'] = $this->provider->specialization;
             $data['bio'] = $this->provider->bio;
        }

        return $data;
    }
}
