<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\DB;
use App\Http\Requests\StoreUserRequest;
use App\Http\Resources\UserResource;

class UserController extends Controller
{
    protected $userService;

    public function __construct(\App\Services\UserService $userService)
    {
        $this->userService = $userService;
    }

    public function index()
    {
        $users = User::with(['roles', 'admin', 'chief', 'subChief', 'employee', 'provider', 'department', 'supervisor'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);
            
        return $this->sendResponse(
            UserResource::collection($users),
            'Users retrieved successfully.',
            $users->currentPage(),
            $users->total(),
            $users->perPage(),
            $users->lastPage(),
            'total_users'
        );
    }

    public function store(StoreUserRequest $request)
    {
        $user = $this->userService->createUser($request->validated());
        // Load relationships explicitly or rely on resource/service to return fully loaded model if needed
        // For simple response, create usually returns just the model. UserResource handles optional stuff.
        return new UserResource($user);
    }

    public function show(string $id)
    {
        $user = User::with(['roles', 'admin', 'chief', 'subChief', 'employee', 'provider', 'department', 'supervisor'])->findOrFail($id);
        return new UserResource($user);
    }

    public function update(StoreUserRequest $request, string $id)
    {
        $user = User::findOrFail($id);
        $user = $this->userService->updateUser($user, $request->validated());
        return new UserResource($user);
    }

    public function destroy(string $id)
    {
        $user = User::findOrFail($id);
        $user->delete();
        return response()->json(['message' => 'User deleted successfully']);
    }
}
