<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;

use App\Http\Requests\StoreSubChiefRequest;
use App\Http\Resources\UserResource;
use App\Models\SubChief;
use Illuminate\Support\Facades\DB;

class SubChiefController extends Controller
{
    protected $userService;

    public function __construct(\App\Services\UserService $userService)
    {
        $this->userService = $userService;
    }

    public function index()
    {
        $subChiefs = User::where('role', 'subchief')->with(['subChief', 'department', 'supervisor'])->paginate(15);
        return $this->sendResponse(
            UserResource::collection($subChiefs),
            'SubChiefs retrieved successfully.',
            $subChiefs->currentPage(),
            $subChiefs->total(),
            $subChiefs->perPage(),
            $subChiefs->lastPage(),
            'total_sub_chiefs'
        );
    }

    public function store(StoreSubChiefRequest $request)
    {
        $data = $request->validated();
        
        // 1. Separate User Data
        $userData = $request->safe()->only([
            'name', 'username', 'email', 'password', 'phone', 
            'address', 'city', 'state', 'zip_code', 'gender', 
            'date_of_birth', 'profile_image', 'department_id', 'supervisor_id'
        ]);
        $userData['role'] = 'subchief'; 

        // 2. Separate SubChief Metadata
        $subChiefData = $request->safe()->only(['badge_number', 'rank']);

        return DB::transaction(function () use ($userData, $subChiefData) {
            $user = $this->userService->createUser($userData);

            // Manual SubChief Creation
            $subChief = $user->subChief;
            if (!$subChief) {
                // Ensure field names match database columns matching fillable: ['user_id', 'practice_id', 'division_id']
                $subChief = SubChief::create([
                    'user_id' => $user->id,
                    ...$subChiefData
                ]);
            } else {
                $subChief->update($subChiefData);
            }
            
            $user->refresh();
            return new UserResource($user);
        });
    }

    public function show(string $id)
    {
        $subChief = SubChief::where('id', $id)->firstOrFail();
        $user = User::where('id', $subChief->user_id)->with(['subChief', 'department', 'supervisor'])->firstOrFail();
        return new UserResource($user);
    }

    public function update(StoreSubChiefRequest $request, string $id)
    {
        $subChiefProfile = SubChief::where('id', $id)->firstOrFail();
        $user = User::where('id', $subChiefProfile->user_id)->firstOrFail();
        
        $userData = $request->safe()->only([
            'name', 'username', 'email', 'password', 'phone', 
            'address', 'city', 'state', 'zip_code', 'gender', 
            'date_of_birth', 'is_active', 'profile_image', 'department_id', 'supervisor_id'
        ]);
        $userData['role'] = 'subchief'; 

        $subChiefData = $request->safe()->only(['badge_number', 'rank']);

        return DB::transaction(function () use ($user, $userData, $subChiefData) {
            $this->userService->updateUser($user, $userData);

            if ($user->subChief) {
                $user->subChief->update($subChiefData);
            } else {
                 SubChief::create([
                    'user_id' => $user->id,
                    ...$subChiefData
                ]);
            }
            
            $user->refresh();
            return new UserResource($user);
        });
    }

    public function destroy(string $id)
    {
        $subChiefProfile = SubChief::where('id', $id)->firstOrFail();
        $user = User::where('id', $subChiefProfile->user_id)->firstOrFail();
        $user->delete();
        return response()->json(['message' => 'SubChief deleted successfully']);
    }
}
