<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;

use App\Http\Requests\StoreProviderRequest;
use App\Http\Resources\UserResource;
use App\Models\Provider;
use App\Services\UserService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class ProviderController extends Controller
{
    protected $userService;

    public function __construct(UserService $userService)
    {
        $this->userService = $userService;
    }

    public function index()
    {
        $providers = User::where('role', 'provider')->with(['provider', 'department', 'supervisor'])->paginate(15);
        return $this->sendResponse(
            UserResource::collection($providers),
            'Providers retrieved successfully.',
            $providers->currentPage(),
            $providers->total(),
            $providers->perPage(),
            $providers->lastPage(),
            'total_providers'
        );
    }

    public function store(StoreProviderRequest $request)
    {
        $data = $request->validated();
        
        $userData = $request->safe()->only([
            'name', 'username', 'email', 'password', 'phone', 
            'address', 'city', 'state', 'zip_code', 'gender', 
            'date_of_birth', 'profile_image', 'department_id', 'supervisor_id'
        ]);
        $userData['role'] = 'provider'; 

        $providerData = $request->safe()->only(['npi', 'license_number', 'specialization', 'bio', 'doxy_me_link']);

        return DB::transaction(function () use ($userData, $providerData) {
            $user = $this->userService->createUser($userData);
            
            $provider = $user->provider;
            if (!$provider) {
                // Auto-set Doxy.me link if not provided
                if (empty($providerData['doxy_me_link'])) {
                    // Fallback to the default gateway if configured, otherwise try to guess or leave null
                    $defaultRoom = config('services.doxy.default_room');
                    
                    if ($defaultRoom) {
                         $providerData['doxy_me_link'] = $defaultRoom;
                    } else {
                        // Fallback logic: Generate slug
                        $roomName = $userData['username'] ?? (Str::slug($userData['name']) . '-' . $user->id);
                        $baseUrl = config('services.doxy.base_url');
                        $baseUrl = rtrim($baseUrl, '/');
                        $providerData['doxy_me_link'] = "{$baseUrl}/{$roomName}";
                    }
                }

                $provider = Provider::create([
                    'user_id' => $user->id,
                    ...$providerData
                ]);
            } else {
                $provider->update($providerData);
            }
            
            $user->refresh();

            return new UserResource($user);
        });
    }

    public function show(string $id)
    {
        // Find Provider by ID (Profile ID), not User ID
        $provider = Provider::where('id', $id)->firstOrFail();
        
        // Retrieve the associated User with necessary relations
        $user = User::where('id', $provider->user_id)
                    ->with(['provider', 'department', 'supervisor'])
                    ->firstOrFail();

        return new UserResource($user);
    }

    public function update(StoreProviderRequest $request, string $id)
    {
        // Find Provider by ID
        $providerProfile = Provider::where('id', $id)->firstOrFail();
        
        // Get User from Provider
        $user = User::where('id', $providerProfile->user_id)->firstOrFail();
        
        $data = $request->validated();
        
        $userData = $request->safe()->only([
            'name', 'username', 'email', 'password', 'phone', 
            'address', 'city', 'state', 'zip_code', 'gender', 
            'date_of_birth', 'is_active', 'profile_image', 'department_id', 'supervisor_id'
        ]);
        $userData['role'] = 'provider'; 

        $providerData = $request->safe()->only(['npi', 'license_number', 'specialization', 'bio', 'doxy_me_link']);

        return DB::transaction(function () use ($user, $userData, $providerData) {
            $this->userService->updateUser($user, $userData);

            if ($user->provider) {
                $user->provider->update($providerData);
            } else {
                 Provider::create([
                    'user_id' => $user->id,
                    ...$providerData
                ]);
            }
            
            $user->refresh();

            return new UserResource($user);
        });
    }

    public function destroy(string $id)
    {
        // Find Provider by ID
        $providerProfile = Provider::where('id', $id)->firstOrFail();
        $user = User::where('id', $providerProfile->user_id)->firstOrFail();
        
        $user->delete(); // This cascades to provider profile usually
        return response()->json(['message' => 'Provider deleted successfully']);
    }
}
