<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\ProviderAvailabilityResource;
use App\Models\Provider;
use App\Models\ProviderAvailability;
use App\Models\ProviderAvailabilitySlot;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Services\AvailabilityService;
use Illuminate\Support\Facades\Log;

class ProviderAvailabilityController extends Controller
{
    // 1. Create the Provider Availability
    public function setAvailability(Request $request)
    {
        try {
            $validated = $request->validate([
                'title' => 'required|string|max:255',
                'slots.*.*.start_time' => 'required|date_format:H:i',
                'slots.*.*.end_time' => 'required|date_format:H:i',
                'type' => 'required|in:in_person,spruce',
                'recurrence' => 'nullable|string|max:255',
            ]);

            DB::beginTransaction();

            // Create the Provider Availability
            $availability = ProviderAvailability::create([
                'provider_id' => $request->provider_id,
                'title' => $request->title,
                'type' => $request->type,
                'recurrence' => $request->recurrence,
            ]);

            // Add Slots without duplicates
            if ($request->has('slots')) {
                foreach ($request->slots as $day => $timeSlots) {
                    foreach ($timeSlots as $slot) {
                        $duplicate = ProviderAvailabilitySlot::where('provider_availability_id', $availability->id)
                            ->where('day_of_week', $day)
                            ->where('start_time', $slot['start_time'])
                            ->where('end_time', $slot['end_time'])
                            ->exists();

                        if (!$duplicate) {
                            ProviderAvailabilitySlot::create([
                                'provider_availability_id' => $availability->id,
                                'day_of_week' => $day,
                                'start_time' => $slot['start_time'],
                                'end_time' => $slot['end_time'],
                            ]);
                        }
                    }
                }
            }
            DB::commit();
            
            return response()->json(['message' => 'Availability created successfully.'], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating provider availability: ' . $e->getMessage());

            return response()->json([
                'message' => 'Failed to create availability.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    // Get all the provider Availability (Filtering based on roles: Provider, Admin, Employee)
    public function getAllProvidersAvailability(Request $request)
    {
        try {
            $user = Auth::user();
            
            if (!$user) {
                 return response()->json(['message' => 'Unauthenticated.'], 401);
            }

            $query = ProviderAvailability::query();

            // Apply filters similar to pathways if needed
            if ($request->filled('search')) {
                $search = $request->search;
                $query->whereHas('provider', function($q) use ($search) {
                    $q->where(function($sub) use ($search) {
                        $sub->whereHas('user', function($userQ) use ($search) {
                             $userQ->where('first_name', 'like', "%{$search}%")
                                   ->orWhere('last_name', 'like', "%{$search}%")
                                   ->orWhere('username', 'like', "%{$search}%");
                        })
                        ->orWhere('specialization', 'like', "%{$search}%");
                    });
                });
            }

            $query->with(['slots', 'provider.user']);

            $availabilities = $query->get();

            return response()->json([
                'success' => true,
                'data' => ProviderAvailabilityResource::collection($availabilities),
                'message' => 'Availabilities retrieved successfully.'
            ]);

        } catch (\Exception $e) {
             return response()->json([
                'success' => false,
                'message' => 'Failed to retrieve availabilities.',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    // Update the provider availability
    public function updateAvailability(Request $request, $id)
    {
        $validated = $request->validate([
            'title' => 'sometimes|required|string|max:255',
            'type' => 'sometimes|required|in:in_person,spruce',
            'recurrence' => 'nullable|string|max:255',
            'slots.*.*.start_time' => 'required_with:slots|date_format:H:i',
            'slots.*.*.end_time' => 'required_with:slots|date_format:H:i',
        ]);

        try {
            $user = Auth::user();
            $providerId = Provider::where('user_id', $user->id)->value('id');
            
            $availability = ProviderAvailability::find($id);
            
            if (!$availability) {
                return response()->json(['message' => 'Availability not found.'], 404);
            }

            // Authorization check
            if ($providerId && $availability->provider_id !== $providerId) {
                 $isAdmin = \App\Models\Admin::where('user_id', $user->id)->exists();
                 if (!$isAdmin) {
                     return response()->json(['message' => 'Unauthorized.'], 403);
                 }
            }

            DB::beginTransaction();

            // Update availability
            $availability->update($request->only(['title', 'type', 'recurrence']));

            // Update slots if provided
            if ($request->has('slots')) {
                // Delete old slots
                ProviderAvailabilitySlot::where('provider_availability_id', $availability->id)->delete();

                // Create new slots
                foreach ($request->slots as $day => $timeSlots) {
                    foreach ($timeSlots as $slot) {
                        ProviderAvailabilitySlot::create([
                            'provider_availability_id' => $availability->id,
                            'day_of_week' => $day,
                            'start_time' => $slot['start_time'],
                            'end_time' => $slot['end_time'],
                        ]);
                    }
                }
            }
            DB::commit();

            return response()->json(['message' => 'Availability updated successfully.']);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Availability update error: ' . $e->getMessage());

            return response()->json([
                'message' => 'Failed to update availability.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function deleteAvailability($id)
    {
        $user = Auth::user();
        $providerId = optional($user->provider)->id;
        $isAdmin = \App\Models\Admin::where('user_id', $user->id)->exists();

        $availability = ProviderAvailability::find($id);

        if (!$availability) {
            return response()->json(['message' => 'Availability not found.'], 404);
        }

        if ($providerId && $availability->provider_id !== $providerId && !$isAdmin) {
             return response()->json(['message' => 'Unauthorized.'], 403);
        }

        DB::beginTransaction();
        try {
            $availability->slots()->delete();
            $availability->delete();
            DB::commit();

            return response()->json(['message' => 'Availability deleted successfully.']);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'Failed to delete availability.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function checkAvailability(Request $request)
    {
        return AvailabilityService::providerAvailabilitybyId();
    }
    
    public function getAvailabilityById($providerId)
    {
        return AvailabilityService::providerAvailabilitybyId($providerId);
    }
}
