<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rules\Password;

class PasswordController extends Controller
{
    /**
     * Change user password (first-time or regular change).
     */
    public function changePassword(Request $request)
    {
        $request->validate([
            'current_password' => 'required|string',
            'new_password' => ['required', 'string', 'confirmed', Password::min(8)->mixedCase()->numbers()],
        ]);

        $user = Auth::user();

        // Verify current password
        if (!Hash::check($request->current_password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'Current password is incorrect.'
            ], 422);
        }

        // Check if new password is same as current
        if (Hash::check($request->new_password, $user->password)) {
            return response()->json([
                'success' => false,
                'message' => 'New password must be different from current password.'
            ], 422);
        }

        // Update password and mark as changed
        $user->update([
            'password' => Hash::make($request->new_password),
            'password_changed' => true
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Password changed successfully. You can now access the system.',
            'data' => [
                'password_changed' => true
            ]
        ]);
    }

    /**
     * Check if user needs to change password.
     */
    public function checkPasswordStatus()
    {
        $user = Auth::user();

        return response()->json([
            'success' => true,
            'data' => [
                'password_changed' => (bool)$user->password_changed,
                'requires_password_change' => !$user->password_changed
            ]
        ]);
    }
}
