<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\IncidentReport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class IncidentReportController extends Controller
{
    /**
     * List incident reports.
     * Filter by user role (Officers see own, Chiefs/Admins see relevant).
     */

    public function index(Request $request)
    {
        $user = Auth::user();
        $query = IncidentReport::with('user');

        // Logic based on Role
        if ($user->role === 'employee') {
            // Employee: See only own reports
            $query->where('user_id', $user->id);

        } elseif ($user->role === 'subchief') {
            // SubChief: See own + reports from employees in their squad (SubDivision)
            $subChiefProfile = $user->subChief; 
            if ($subChiefProfile && $subChiefProfile->subDivision) {
                 $subDivisionId = $subChiefProfile->subDivision->id;
                 
                 $query->where(function($q) use ($user, $subDivisionId) {
                     // Own reports
                     $q->where('user_id', $user->id) 
                       // OR reports from employees in managed SubDivision
                       ->orWhereHas('user.employee', function($sq) use ($subDivisionId) {
                           $sq->where('sub_division_id', $subDivisionId);
                       });
                 });
            } else {
                // If not assigned to a squad, fallback to only own reports
                $query->where('user_id', $user->id);
            }

        } elseif ($user->role === 'chief') {
             // Chief: See own + reports from all employees/subchiefs in their Division
             $chiefProfile = $user->chief;
             if ($chiefProfile && $chiefProfile->division) {
                 $divisionId = $chiefProfile->division->id;
                 
                 $query->where(function($q) use ($user, $divisionId) {
                     $q->where('user_id', $user->id)
                       // Find users linked to this practice's division via SubDivisions
                       ->orWhereHas('user.employee.subDivision', function($sq) use ($divisionId) {
                           $sq->where('division_id', $divisionId);
                       })
                       // OR SubChiefs in this division
                       ->orWhereHas('user.subChief', function($sq) use ($divisionId) {
                           $sq->where('division_id', $divisionId);
                       });
                 });
             } else {
                 $query->where('user_id', $user->id);
             }
        }
        // Admins/Providers see all (default query left as is)

        $reports = $query->orderBy('created_at', 'desc')->paginate(20);

        return $this->sendResponse(
            $reports->items(),
            'Incident reports retrieved successfully.',
            $reports->currentPage(),
            $reports->total(),
            $reports->perPage(),
            $reports->lastPage(),
            'total_reports'
        );
    }

    /**
     * Submit a new Incident Report (Injury or Exposure).
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'form_type' => ['required', Rule::in(['injury', 'exposure'])],
            
            // Common Fields
            'incident_date' => 'required|date',
            'incident_time' => 'required',
            'location' => 'required|string',
            'district' => 'required|string',
            'consent_telehealth' => 'accepted', // Must be true
            
            // Injury Specific
            'injury_category' => 'required_if:form_type,injury|array',
            'mechanism_description' => 'required_if:form_type,injury|string',
            'body_parts' => 'required_if:form_type,injury|array',
            'red_flags' => 'nullable|array',
            'has_red_flags' => 'boolean',
            'pain_score' => 'nullable|integer|min:0|max:10',
            
            // Exposure Specific
            'exposure_type' => 'required_if:form_type,exposure|string',
            'exposure_category' => 'nullable|array',
            'source_status' => 'nullable|string',
            
            // Attestation
            'officer_attestation' => 'accepted',
        ]);

        // Auto-fill user info
        $data = $request->all();
        $data['user_id'] = Auth::id();
        $data['attested_at'] = now();
        $data['status'] = 'submitted';

        // Check for Red Flags causing Immediate Action
        if (!empty($data['red_flags']) && !in_array('None of the above', $data['red_flags'])) {
            $data['has_red_flags'] = true;
            $data['status'] = 'urgent_attention';
        }

        $report = IncidentReport::create($data);

        return response()->json([
            'message' => 'Incident report submitted successfully.',
            'report' => $report
        ], 201);
    }

    /**
     * View a specific report.
     */
    public function show($id)
    {
        $report = IncidentReport::with('user')->findOrFail($id);
        
        // Authorization check
        if (Auth::user()->role === 'employee' && $report->user_id !== Auth::id()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        return response()->json($report);
    }

    /**
     * Update an incident report.
     */
    public function update(Request $request, $id)
    {
        $report = IncidentReport::findOrFail($id);
        
        // Authorization: Only admin or owner can update (add specific logic as needed)
        if (Auth::user()->role === 'employee' && $report->user_id !== Auth::id()) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $report->update($request->all());

        return response()->json([
            'message' => 'Incident report updated successfully.',
            'report' => $report
        ]);
    }

    /**
     * Delete an incident report.
     */
    public function destroy($id)
    {
        $report = IncidentReport::findOrFail($id);
        
        // Authorization
        if (Auth::user()->role === 'employee' && $report->user_id !== Auth::id()) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $report->delete();

        return response()->json(['message' => 'Incident report deleted successfully.']);
    }
}
