<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;

use App\Http\Requests\StoreEmployeeRequest;
use App\Http\Resources\UserResource;
use App\Models\Employee;
use Illuminate\Support\Facades\DB;

class EmployeeController extends Controller
{
    protected $userService;

    public function __construct(\App\Services\UserService $userService)
    {
        $this->userService = $userService;
    }

    public function index()
    {
        $employees = User::where('role', 'employee')->with(['employee', 'department', 'supervisor'])->paginate(15);
        return $this->sendResponse(
            UserResource::collection($employees),
            'Employees retrieved successfully.',
            $employees->currentPage(),
            $employees->total(),
            $employees->perPage(),
            $employees->lastPage(),
            'total_employees'
        );
    }

    public function store(StoreEmployeeRequest $request)
    {
        $data = $request->validated();
        
        // 1. Separate User Data
        $userData = $request->safe()->only([
            'name', 'username', 'email', 'password', 'phone', 
            'address', 'city', 'state', 'zip_code', 'gender', 
            'date_of_birth', 'profile_image', 'department_id', 'supervisor_id'
        ]);
        $userData['role'] = 'employee'; 

        // 2. Separate Employee Metadata
        $employeeData = $request->safe()->only(['badge_number', 'rank', 'position', 'hired_date']);

        return DB::transaction(function () use ($userData, $employeeData) {
            $user = $this->userService->createUser($userData);

            // Manual Employee Creation
            $employee = $user->employee;
            if (!$employee) {
                $employee = Employee::create([
                    'user_id' => $user->id,
                    ...$employeeData
                ]);
            } else {
                $employee->update($employeeData);
            }
            
            $user->refresh();
            return new UserResource($user);
        });
    }

    public function show(string $id)
    {
        $employee = Employee::where('id', $id)->firstOrFail();
        $user = User::where('id', $employee->user_id)->with(['employee', 'department', 'supervisor'])->firstOrFail();
        return new UserResource($user);
    }

    public function update(StoreEmployeeRequest $request, string $id)
    {
        $employeeProfile = Employee::where('id', $id)->firstOrFail();
        $user = User::where('id', $employeeProfile->user_id)->firstOrFail();
        
        $userData = $request->safe()->only([
            'name', 'username', 'email', 'password', 'phone', 
            'address', 'city', 'state', 'zip_code', 'gender', 
            'date_of_birth', 'is_active', 'profile_image', 'department_id', 'supervisor_id'
        ]);
        $userData['role'] = 'employee'; 

        $employeeData = $request->safe()->only(['badge_number', 'rank', 'position', 'hired_date']);

        return DB::transaction(function () use ($user, $userData, $employeeData) {
            $this->userService->updateUser($user, $userData);

            if ($user->employee) {
                $user->employee->update($employeeData);
            } else {
                 Employee::create([
                    'user_id' => $user->id,
                    ...$employeeData
                ]);
            }
            
            $user->refresh();
            return new UserResource($user);
        });
    }

    public function destroy(string $id)
    {
        $employeeProfile = Employee::where('id', $id)->firstOrFail();
        $user = User::where('id', $employeeProfile->user_id)->firstOrFail();
        $user->delete();
        return response()->json(['message' => 'Employee deleted successfully']);
    }
}
