<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;

use App\Http\Requests\StoreChiefRequest;
use App\Http\Resources\UserResource;
use App\Models\Chief;
use Illuminate\Support\Facades\DB;

class ChiefController extends Controller
{
    protected $userService;

    public function __construct(\App\Services\UserService $userService)
    {
        $this->userService = $userService;
    }

    public function index()
    {
        $chiefs = User::where('role', 'chief')->with(['chief', 'department', 'supervisor'])->paginate(15);
        return $this->sendResponse(
            UserResource::collection($chiefs),
            'Chiefs retrieved successfully.',
            $chiefs->currentPage(),
            $chiefs->total(),
            $chiefs->perPage(),
            $chiefs->lastPage(),
            'total_chiefs'
        );
    }

    public function store(StoreChiefRequest $request)
    {
        $data = $request->validated();
        
        // 1. Separate User Data
        $userData = $request->safe()->only([
            'name', 'username', 'email', 'password', 'phone', 
            'address', 'city', 'state', 'zip_code', 'gender', 
            'date_of_birth', 'profile_image', 'department_id', 'supervisor_id'
        ]);
        $userData['role'] = 'chief'; 

        // 2. Separate Chief Metadata
        $chiefData = $request->safe()->only(['badge_number', 'rank', 'appointed_date']);

        return DB::transaction(function () use ($userData, $chiefData) {
            $user = $this->userService->createUser($userData);

            // Manual Chief Creation
            $chief = $user->chief;
            if (!$chief) {
                $chief = Chief::create([
                    'user_id' => $user->id,
                    ...$chiefData
                ]);
            } else {
                $chief->update($chiefData);
            }
            
            $user->refresh();
            return new UserResource($user);
        });
    }

    public function show(string $id)
    {
        $chief = Chief::where('id', $id)->firstOrFail();
        $user = User::where('id', $chief->user_id)->with(['chief', 'department', 'supervisor'])->firstOrFail();
        return new UserResource($user);
    }

    public function update(StoreChiefRequest $request, string $id)
    {
        $chiefProfile = Chief::where('id', $id)->firstOrFail();
        $user = User::where('id', $chiefProfile->user_id)->firstOrFail();
        
        $userData = $request->safe()->only([
            'name', 'username', 'email', 'password', 'phone', 
            'address', 'city', 'state', 'zip_code', 'gender', 
            'date_of_birth', 'is_active', 'profile_image', 'department_id', 'supervisor_id'
        ]);
        $userData['role'] = 'chief'; 
        $chiefData = $request->safe()->only(['badge_number', 'rank', 'appointed_date']);

        return DB::transaction(function () use ($user, $userData, $chiefData) {
            $this->userService->updateUser($user, $userData);

            if ($user->chief) {
                $user->chief->update($chiefData);
            } else {
                 Chief::create([
                    'user_id' => $user->id,
                    ...$chiefData
                ]);
            }
            
            $user->refresh();
            return new UserResource($user);
        });
    }

    public function destroy(string $id)
    {
        $chiefProfile = Chief::where('id', $id)->firstOrFail();
        $user = User::where('id', $chiefProfile->user_id)->firstOrFail();
        $user->delete();
        return response()->json(['message' => 'Chief deleted successfully']);
    }
}
