<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\LoginLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use App\Services\ImageService; // Assuming we have this service or will need to stub it
use App\Http\Resources\UserResource; 

class AuthController extends Controller
{
    public function login(Request $request)
    {
        // Support 'email', 'username', or 'login' field in request
        $loginField = 'email'; // Default assumption
        if ($request->has('username')) $loginField = 'username';
        if ($request->has('login')) $loginField = 'login'; // Generic field

        $validator = Validator::make($request->all(), [
            $loginField => 'required',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => 'Validation Error.', 'message' => $validator->errors()], 422);
        }

        $loginInput = $request->input($loginField);
        // Determine if input is email or username
        $authColumn = filter_var($loginInput, FILTER_VALIDATE_EMAIL) ? 'email' : 'username';

        if (Auth::attempt([$authColumn => $loginInput, 'password' => $request->password])) {
            $user = User::with('roles')->find(Auth::id());
            
            // 🚨 Check if user is active
            if (!$user->is_active) {
                Auth::logout();
                return response()->json(['error' => 'Account Inactive.', 'message' => 'Please contact admin, account not active.'], 403);
            }

            // Create Login Log
            $log = LoginLog::create([
                'user_id' => $user->id,
                'login_time' => now(),
                'ip_address' => $request->ip(),
                'user_agent' => $request->userAgent(),
            ]);
            
            $user->current_login_log_id = $log->id;
            
            // Login Count Logic (simplified as requested, just increment)
            $user->daily_login_count = ($user->daily_login_count ?? 0) + 1;
            $user->last_login_at = now();
            $user->save();

            $token = $user->createToken('auth_token')->plainTextToken;

            // 1. Prepare Role Data & IDs
            $roleNames = $user->roles->pluck('name')->toArray();
            
            $roleIds = [];
            if ($user->admin) $roleIds['admin_id'] = $user->admin->id;
            if ($user->chief) $roleIds['chief_id'] = $user->chief->id;
            if ($user->subChief) $roleIds['sub_chief_id'] = $user->subChief->id;
            if ($user->employee) $roleIds['employee_id'] = $user->employee->id;
            if ($user->provider) $roleIds['provider_id'] = $user->provider->id;
            // Add patient check if relation exists
            // if ($user->patient) $roleIds['patient_id'] = $user->patient->id;

            // 2. Profile Completion Check
            $profilesCompleted = [];
            
            $checkCompletion = function($resourceClass, $modelId) use ($request) {
                if (!$modelId) return null;
                $modelClass = null;
                // Map Resources to Models
                if (str_contains($resourceClass, 'Provider')) $modelClass = \App\Models\Provider::class;
                if (str_contains($resourceClass, 'Patient')) $modelClass = \App\Models\Patient::class; 
                // Add Admin/Employee/etc if they have resources & completion logic
                // For now, only Provider/Patient have it in our code.
                
                if ($modelClass) {
                   $model = $modelClass::find($modelId);
                   if ($model) {
                       $resource = new $resourceClass($model);
                       $dataArr = $resource->toArray($request);
                       return empty($dataArr['missing_values']);
                   }
                }
                return true; // Default to true if no specific logic
            };

            // Check specific roles
            if (isset($roleIds['provider_id'])) {
                $profilesCompleted['provider'] = $checkCompletion(\App\Http\Resources\ProviderResource::class, $roleIds['provider_id']);
            }
            if (isset($roleIds['patient_id'])) {
                $profilesCompleted['patient'] = $checkCompletion(\App\Http\Resources\PatientResource::class, $roleIds['patient_id']);
            }

            // Fill missing roles with true
            foreach ($roleNames as $r) {
                if (!isset($profilesCompleted[$r])) {
                    $profilesCompleted[$r] = true; 
                }
            }

            // 3. Update User Profile Completed Status
            // Check primary role status. Logic: if user->role matches one of the checked ones, use that.
            // fallback: check if ALL are complete? or just primary.
            $primaryRoleComplete = true;
            if (isset($profilesCompleted[$user->role])) {
                $primaryRoleComplete = $profilesCompleted[$user->role];
            }
            
            if ((bool)$user->profile_completed !== $primaryRoleComplete) {
                $user->profile_completed = $primaryRoleComplete;
                $user->save();
            }

            $data = [
                'access_token' => $token,
                'token_type' => 'Bearer',
                'user_role' => $user->role, 
                'roles' => $roleNames,
                'full_name' => $user->name,
                'login_count_today' => $user->daily_login_count,
                'main_id' => $user->id,
                ...$roleIds,
                'main_profile_completed' => (bool)$user->profile_completed,
                'profiles_completed' => $profilesCompleted,
                'password_changed' => (bool)$user->password_changed,
                'requires_password_change' => !$user->password_changed,
            ];

            return response()->json(['success' => true, 'data' => $data, 'message' => 'User login successfully.']);

        } else {
            return response()->json(['error' => 'Unauthorised.', 'message' => 'Incorrect email and password'], 401);
        }
    }

    public function logout(Request $request)
    {
        $user = Auth::user(); // or $request->user()
        
        if ($user) {
            $logId = $user->current_login_log_id;

            if ($logId) {
                $log = LoginLog::find($logId);
                if ($log && !$log->logout_time) {
                    $log->logout_time = now();
                    $log->session_duration = $log->login_time->diffInSeconds(now());
                    $log->save();
                }
                $user->current_login_log_id = null;
                $user->save();
            }
            
            $user->currentAccessToken()->delete();
        }

        return response()->json(['success' => true, 'message' => 'Successful Logout']);
    }

    // Forgot Password Flow

    public function forgotPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|exists:users,email',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors(),
            ], 422);
        }

        $status = \Illuminate\Support\Facades\Password::sendResetLink(
            $request->only('email')
        );
        
        if ($status === \Illuminate\Support\Facades\Password::RESET_LINK_SENT) {
            return response()->json(['message' => __($status)], 200);
        }

        return response()->json(['message' => __($status)], 400);
    }

    public function resetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email|exists:users,email',
            'token' => 'required',
            'password' => 'required|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'errors' => $validator->errors(),
            ], 422);
        }

        $status = \Illuminate\Support\Facades\Password::reset(
            $request->only('email', 'password', 'password_confirmation', 'token'),
            function ($user, $password) {
                $user->forceFill([
                    'password' => Hash::make($password),
                    'remember_token' => \Illuminate\Support\Str::random(60),
                ])->save();
            }
        );

        if ($status === \Illuminate\Support\Facades\Password::PASSWORD_RESET) {
            return response()->json(['message' => __($status)], 200);
        } else {
            return response()->json(['message' => __($status)], 400);
        }
    }

    public function validateResetToken(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:users,email',
            'token' => 'required'
        ]);

        $record = \Illuminate\Support\Facades\DB::table('password_reset_tokens')
            ->where('email', $request->email)
            ->first();

        if (!$record) {
            return response()->json(['message' => 'Reset token not found or invalid.'], 404);
        }

        // Validate token (Basic check, Laravel hashes tokens in newer versions, assuming default DB token table here)
        // If your table stores plain tokens (default for older Laravel auth), simple check.
        // If hashed, utilize PasswordBroker::tokenExists logic.

        return response()->json(['message' => 'Token is valid'], 200);
    }
}
