<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Appointment;
use App\Http\Resources\AppointmentResource;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class AppointmentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $query = Appointment::with(['employee.user', 'provider.user']);

        if ($user->provider) {
             $query->where('provider_id', $user->provider->id);
        } elseif ($user->employee) {
             $query->where('employee_id', $user->employee->id);
        }
       
        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('appointment_date', [$request->start_date, $request->end_date]);
        }

        $appointments = $query->paginate(15);

        return $this->sendResponse(
            AppointmentResource::collection($appointments),
            'Appointments retrieved successfully.',
            $appointments->currentPage(),
            $appointments->total(),
            $appointments->perPage(),
            $appointments->lastPage(),
            'total_appointments'
        );
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'employee_id' => 'required|exists:employees,id',
            'provider_id' => 'required|exists:providers,id',
            'appointment_date' => 'required|date',
            'start_time' => 'required',
            'end_time' => 'required',
            'title' => 'nullable|string',
            'description' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            $appointment = Appointment::create([
                'employee_id' => $request->employee_id,
                'provider_id' => $request->provider_id,
                'appointment_date' => $request->appointment_date,
                'start_time' => $request->start_time,
                'end_time' => $request->end_time,
                'title' => $request->title ?? 'New Appointment',
                'description' => $request->description,
                'type' => 'telehealth', 
                'status' => 'scheduled',
                'cpt_code' => $request->cpt_code,
                'fees' => $request->fees,
            ]);

            DB::commit();

            return new AppointmentResource($appointment->load(['employee.user', 'provider.user']));

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Appointment creation failed: ' . $e->getMessage());
            return response()->json(['message' => 'Failed to create appointment', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $appointment = Appointment::with(['employee.user', 'provider.user'])->findOrFail($id);
        
        // Security check (basic)
        $user = Auth::user();
        if ($user->provider && $appointment->provider_id !== $user->provider->id) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }
        if ($user->employee && $appointment->employee_id !== $user->employee->id) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }
        
        return new AppointmentResource($appointment);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $appointment = Appointment::findOrFail($id);

        try {
            $appointment->update($request->all());
            return new AppointmentResource($appointment->load(['employee.user', 'provider.user']));
        } catch (\Exception $e) {
            return response()->json(['message' => 'Failed to update appointment', 'error' => $e->getMessage()], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $appointment = Appointment::findOrFail($id);
        $appointment->delete();
        return response()->json(['message' => 'Appointment deleted successfully']);
    }

    /**
     * Securely join the appointment.
     */
    public function join(string $id)
    {
        $appointment = Appointment::with('provider')->findOrFail($id);
        $user = Auth::user();

        $authorized = false;
        
        if ($user->provider && $appointment->provider_id === $user->provider->id) {
             $authorized = true;
        } elseif ($user->employee && $appointment->employee_id === $user->employee->id) {
             $authorized = true;
        } elseif ($user->hasRole && $user->hasRole('admin')) { // Assuming generic admin role for now
             $authorized = true;
        }

        if (!$authorized) {
            abort(403, 'Unauthorized access to this appointment.');
        }

        // 2. Timeline Check (Optional: Strict mode)
        // Allowing join 10 minutes early.
        // if (now()->lt(\Carbon\Carbon::parse($appointment->appointment_date . ' ' . $appointment->start_time)->subMinutes(10))) {
        //     abort(403, 'It is too early to join this appointment.');
        // }

        $link = $appointment->provider?->doxy_me_link;

        if (!$link) {
            abort(404, 'Telehealth link not available for this provider.');
        }

        // 3. Audit Logging (Optional but recommended)
        // Log::info("User {$user->id} joined appointment {$appointment->id} at " . now());

        return redirect()->away($link);
    }
    
    // Custom Endpoints for Employees (acting as patients) and Providers
    public function employee_appointment_details($employeeId)
    {
        // Check authorization (Employee can only see own, Admin sees all, etc.)
        // For simplicity, just fetching for now.
        $appointments = Appointment::where('employee_id', $employeeId)
            ->with(['provider.user', 'employee.user'])
            ->orderBy('appointment_date', 'desc')
            ->orderBy('start_time', 'desc')
            ->get();
            
        return $this->sendResponse(
            AppointmentResource::collection($appointments),
            'Employee appointments retrieved successfully.'
        );
    }

    public function employee_appointment_summary($employeeId)
    {
        try {
            $appointments = Appointment::where('employee_id', $employeeId)->get();

            if ($appointments->isEmpty()) {
                return $this->sendResponse([
                    'summary' => [
                        'total' => 0,
                        'statuses' => []
                    ]
                ], 'No appointments found for this employee.');
            }

            $summary = [
                'total' => $appointments->count(),
                'statuses' => $appointments->groupBy('status')->map(fn($group) => $group->count())
            ];

            return $this->sendResponse(['summary' => $summary], 'Appointment summary retrieved successfully.');
        } catch (\Exception $e) {
            return $this->sendError('Failed to fetch appointment summary.', ['error' => $e->getMessage()], 500);
        }
    }

    public function provider_appointment_details($providerId)
    {
         $appointments = Appointment::where('provider_id', $providerId)
            ->with(['employee.user', 'provider.user'])
            ->orderBy('appointment_date', 'desc')
            ->orderBy('start_time', 'desc')
            ->get();
            
        return $this->sendResponse(
            AppointmentResource::collection($appointments),
            'Provider appointments retrieved successfully.'
        );
    }
}
