<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;

use App\Http\Requests\StoreAdminRequest;
use App\Http\Resources\UserResource;
use App\Models\Admin;
use Illuminate\Support\Facades\DB;
use App\Services\UserService;

class AdminController extends Controller
{
    protected $userService;

    public function __construct(UserService $userService)
    {
        $this->userService = $userService;
    }

    public function index()
    {
        $admins = User::where('role', 'admin')->with('admin')->paginate(15);
        return $this->sendResponse(
            UserResource::collection($admins),
            'Admins retrieved successfully.',
            $admins->currentPage(),
            $admins->total(),
            $admins->perPage(),
            $admins->lastPage(),
            'total_admins'
        );
    }

    public function store(StoreAdminRequest $request)
    {
        $data = $request->validated();
        
        // 1. Separate User Data
        $userData = $request->safe()->only([
            'name', 'username', 'email', 'password', 'phone', 
            'address', 'city', 'state', 'zip_code', 'gender', 
            'date_of_birth', 'profile_image'
        ]);
        $userData['role'] = 'admin';

        return DB::transaction(function () use ($userData) {
            $user = $this->userService->createUser($userData);

            // Manual Admin Creation
            $admin = $user->admin;
            if (!$admin) {
                Admin::create(['user_id' => $user->id]);
            }
            
            $user->refresh();
            return new UserResource($user);
        });
    }

    public function show(string $id)
    {
        $admin = Admin::where('id', $id)->firstOrFail();
        $user = User::where('id', $admin->user_id)->with('admin')->firstOrFail();
        return new UserResource($user);
    }

    public function update(StoreAdminRequest $request, string $id)
    {
        $adminProfile = Admin::where('id', $id)->firstOrFail();
        $user = User::where('id', $adminProfile->user_id)->firstOrFail();
        
        $userData = $request->safe()->only([
            'name', 'username', 'email', 'password', 'phone', 
            'address', 'city', 'state', 'zip_code', 'gender', 
            'date_of_birth', 'is_active', 'profile_image'
        ]);
        $userData['role'] = 'admin';

        return DB::transaction(function () use ($user, $userData) {
            $this->userService->updateUser($user, $userData);
            
            if (!$user->admin) {
                Admin::create(['user_id' => $user->id]);
            }
            
            $user->refresh();
            return new UserResource($user);
        });
    }

    public function destroy(string $id)
    {
        $adminProfile = Admin::where('id', $id)->firstOrFail();
        $user = User::where('id', $adminProfile->user_id)->firstOrFail();
        $user->delete();
        return response()->json(['message' => 'Admin deleted successfully']);
    }
}
